<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    /**
     * Create an index and ignore duplicate errors safely.
     */
    private function createIndexSafe(string $table, array $columns, string $name): void
    {
        try {
            Schema::table($table, function (Blueprint $blueprint) use ($columns, $name) {
                $blueprint->index($columns, $name);
            });
        } catch (\Throwable $e) {
            // Ignore duplicate index or missing table errors to keep migration idempotent across envs
        }
    }

    public function up(): void
    {
        // Transactions
        $this->createIndexSafe('transactions', ['business_id', 'type', 'status'], 'trx_biz_type_status_idx');
        $this->createIndexSafe('transactions', ['business_id', 'type', 'location_id'], 'trx_biz_type_loc_idx');
        $this->createIndexSafe('transactions', ['business_id', 'type', 'transaction_date'], 'trx_biz_type_tdate_idx');
        $this->createIndexSafe('transactions', ['business_id', 'type', 'delivery_date'], 'trx_biz_type_ddate_idx');
        $this->createIndexSafe('transactions', ['approval_status'], 'trx_approval_status_idx');
        $this->createIndexSafe('transactions', ['created_by'], 'trx_created_by_idx');

        // Purchase lines
        $this->createIndexSafe('purchase_lines', ['transaction_id'], 'pl_transaction_id_idx');
        $this->createIndexSafe('purchase_lines', ['product_id'], 'pl_product_id_idx');
        $this->createIndexSafe('purchase_lines', ['variation_id'], 'pl_variation_id_idx');
        $this->createIndexSafe('purchase_lines', ['purchase_requisition_line_id'], 'pl_pr_line_id_idx');

        // Variations
        $this->createIndexSafe('variations', ['product_variation_id'], 'var_prod_variation_id_idx');
        $this->createIndexSafe('variations', ['deleted_at'], 'var_deleted_at_idx');

        // Variation Location Details
        $this->createIndexSafe('variation_location_details', ['variation_id', 'location_id'], 'vld_variation_location_idx');

        // Products
        $this->createIndexSafe('products', ['business_id', 'enable_stock', 'is_inactive'], 'prod_biz_stock_active_idx');
        $this->createIndexSafe('products', ['alert_quantity'], 'prod_alert_qty_idx');
        $this->createIndexSafe('products', ['category_id'], 'prod_category_id_idx');
        $this->createIndexSafe('products', ['brand_id'], 'prod_brand_id_idx');
        $this->createIndexSafe('products', ['sku'], 'prod_sku_idx');
        $this->createIndexSafe('products', ['name'], 'prod_name_idx');

        // Transaction sell lines
        $this->createIndexSafe('transaction_sell_lines', ['transaction_id'], 'tsl_transaction_id_idx');
        $this->createIndexSafe('transaction_sell_lines', ['product_id'], 'tsl_product_id_idx');
        $this->createIndexSafe('transaction_sell_lines', ['variation_id'], 'tsl_variation_id_idx');
        $this->createIndexSafe('transaction_sell_lines', ['parent_sell_line_id'], 'tsl_parent_line_id_idx');

        // Transaction payments
        $this->createIndexSafe('transaction_payments', ['transaction_id'], 'tp_transaction_id_idx');
        $this->createIndexSafe('transaction_payments', ['payment_ref_no'], 'tp_payment_ref_no_idx');
        $this->createIndexSafe('transaction_payments', ['method'], 'tp_method_idx');
    }

    public function down(): void
    {
        // No-op: leaving indexes in place on rollback to avoid downtime.
    }
};

